<?php
require_once __DIR__ . '/../lib/bootstrap.php';
require_role(['ADMIN']);

$pdo = db();

function table_exists(PDO $pdo, string $name): bool {
  $st = $pdo->prepare("SHOW TABLES LIKE ?");
  $st->execute([$name]);
  return (bool)$st->fetchColumn();
}
function column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SHOW COLUMNS FROM `$table` LIKE ?");
  $st->execute([$col]);
  return (bool)$st->fetchColumn();
}

$hasItems = table_exists($pdo, 'items');
if (!$hasItems) {
  include __DIR__ . '/../partials/header.php';
  echo '<div class="alert alert-danger shadow-sm">Table <code>items</code> not found.</div>';
  include __DIR__ . '/../partials/footer.php';
  exit;
}

$hasCategoriesTbl = table_exists($pdo,'categories');
$hasCategoryId = column_exists($pdo,'items','category_id');
$hasCategories = $hasCategoriesTbl && $hasCategoryId;

$hasName = column_exists($pdo,'items','name');
$hasUnit = column_exists($pdo,'items','unit');
$hasStock = column_exists($pdo,'items','stock_qty');
$hasActive = column_exists($pdo,'items','is_active');
$hasCreatedAt = column_exists($pdo,'items','created_at');
$hasUpdatedAt = column_exists($pdo,'items','updated_at');
$hasThreshold = column_exists($pdo,'items','threshold_qty');

$thresholdWarning = !$hasThreshold;

// ---------- Create / Update ----------
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  try {
    $action = $_POST['action'] ?? '';

    if ($action === 'create') {
      $name = trim($_POST['name'] ?? '');
      $unit = trim($_POST['unit'] ?? 'pcs');
      $category_id = (int)($_POST['category_id'] ?? 0);
      $threshold_qty = isset($_POST['threshold_qty']) ? (int)$_POST['threshold_qty'] : null;

      if (!$hasName || $name === '') {
        flash_set('danger','Item name is required.');
        header('Location: ' . base_url('admin/items.php'));
        exit;
      }
      if ($hasThreshold && $threshold_qty !== null && $threshold_qty < 0) $threshold_qty = 0;

      // Uniqueness rule: same name cannot repeat in same category (if categories exist)
      if ($hasCategories) {
        $chk = $pdo->prepare("SELECT id FROM items WHERE name=? AND category_id <=> ? LIMIT 1");
        $chk->execute([$name, $category_id ?: null]);
      } else {
        $chk = $pdo->prepare("SELECT id FROM items WHERE name=? LIMIT 1");
        $chk->execute([$name]);
      }
      if ($chk->fetch()) {
        flash_set('warning','This item name already exists in the selected category.');
        header('Location: ' . base_url('admin/items.php'));
        exit;
      }

      // Build INSERT dynamically based on existing columns
      $cols = [];
      $vals = [];
      $params = [];

      $cols[] = 'name'; $vals[] = '?'; $params[] = $name;

      if ($hasUnit) { $cols[] = 'unit'; $vals[] = '?'; $params[] = $unit; }

      if ($hasCategories) { $cols[] = 'category_id'; $vals[] = '?'; $params[] = ($category_id ?: null); }

      if ($hasStock) { $cols[] = 'stock_qty'; $vals[] = '?'; $params[] = 0; }

      if ($hasThreshold) { $cols[] = 'threshold_qty'; $vals[] = '?'; $params[] = $threshold_qty; }

      if ($hasActive) { $cols[] = 'is_active'; $vals[] = '1'; }

      if ($hasCreatedAt) { $cols[] = 'created_at'; $vals[] = 'NOW()'; }

      $sql = "INSERT INTO items(" . implode(',', $cols) . ") VALUES(" . implode(',', $vals) . ")";
      $stmt = $pdo->prepare($sql);
      $stmt->execute($params);

      flash_set('success','✅ Item added successfully.');
      header('Location: ' . base_url('admin/items.php'));
      exit;
    }

    if ($action === 'update') {
      $id = (int)($_POST['id'] ?? 0);
      $name = trim($_POST['name'] ?? '');
      $unit = trim($_POST['unit'] ?? 'pcs');
      $category_id = (int)($_POST['category_id'] ?? 0);
      $is_active = (int)($_POST['is_active'] ?? 1);
      $threshold_qty = isset($_POST['threshold_qty']) ? (int)$_POST['threshold_qty'] : null;

      if ($id <= 0 || !$hasName || $name === '') {
        flash_set('danger','Invalid item.');
        header('Location: ' . base_url('admin/items.php'));
        exit;
      }
      if ($hasThreshold && $threshold_qty !== null && $threshold_qty < 0) $threshold_qty = 0;

      // Uniqueness rule
      if ($hasCategories) {
        $chk = $pdo->prepare("SELECT id FROM items WHERE name=? AND category_id <=> ? AND id<>? LIMIT 1");
        $chk->execute([$name, $category_id ?: null, $id]);
      } else {
        $chk = $pdo->prepare("SELECT id FROM items WHERE name=? AND id<>? LIMIT 1");
        $chk->execute([$name, $id]);
      }
      if ($chk->fetch()) {
        flash_set('warning','This item name already exists in the selected category.');
        header('Location: ' . base_url('admin/items.php'));
        exit;
      }

      // Build UPDATE dynamically
      $set = [];
      $params = [];

      $set[] = "name=?"; $params[] = $name;
      if ($hasUnit) { $set[] = "unit=?"; $params[] = $unit; }
      if ($hasCategories) { $set[] = "category_id=?"; $params[] = ($category_id ?: null); }
      if ($hasActive) { $set[] = "is_active=?"; $params[] = $is_active; }
      if ($hasThreshold) { $set[] = "threshold_qty=?"; $params[] = $threshold_qty; }
      if ($hasUpdatedAt) { $set[] = "updated_at=NOW()"; }

      $params[] = $id;

      $sql = "UPDATE items SET " . implode(', ', $set) . " WHERE id=?";
      $stmt = $pdo->prepare($sql);
      $stmt->execute($params);

      flash_set('success','✅ Item updated successfully.');
      header('Location: ' . base_url('admin/items.php'));
      exit;
    }

  } catch (Throwable $e) {
    flash_set('danger', 'Error: ' . $e->getMessage());
    header('Location: ' . base_url('admin/items.php'));
    exit;
  }
}

// ---------- Fetch categories ----------
$categories = [];
if ($hasCategories) {
  $categories = $pdo->query("SELECT id, name FROM categories ORDER BY name")->fetchAll();
}

// ---------- Fetch items ----------
$sqlItems = "
  SELECT i.*
  " . ($hasCategories ? ", c.name AS category_name" : "") . "
  FROM items i
  " . ($hasCategories ? "LEFT JOIN categories c ON c.id=i.category_id" : "") . "
  ORDER BY " . ($hasCategories ? "c.name, " : "") . " i.name
";
$items = $pdo->query($sqlItems)->fetchAll();

// ---------- Low stock list ----------
$lowStock = [];
if ($hasThreshold && $hasStock) {
  foreach ($items as $it) {
    $th = $it['threshold_qty'];
    if ($th !== null && (int)$th > 0 && (int)$it['stock_qty'] <= (int)$th) {
      $lowStock[] = $it;
    }
  }
}

include __DIR__ . '/../partials/header.php';
?>

<div class="d-flex justify-content-between align-items-start mb-3">
  <div>
    <h3 class="mb-1">Items</h3>
    <div class="text-muted">Low-stock threshold (alert when stock ≤ threshold)</div>
  </div>
</div>

<?php if ($thresholdWarning): ?>
  <div class="alert alert-warning shadow-sm">
    <strong>Threshold is not enabled yet.</strong>
    Run <code>sql/add_item_threshold.sql</code> to add <code>items.threshold_qty</code>.
  </div>
<?php endif; ?>

<?php if ($hasThreshold && count($lowStock) > 0): ?>
  <div class="alert alert-danger shadow-sm d-flex justify-content-between align-items-center">
    <div>
      <strong>⚠ Low stock:</strong> <?=h(count($lowStock))?> item(s) are at or below threshold.
      <span class="text-muted">Please purchase to refill.</span>
    </div>
    <a class="btn btn-sm btn-outline-light" href="#low-stock">View</a>
  </div>
<?php endif; ?>

<div class="card shadow-sm mb-3">
  <div class="card-body">
    <h5 class="mb-3">Add new item</h5>
    <form method="post" class="row g-2 align-items-end">
      <input type="hidden" name="action" value="create">

      <?php if ($hasCategories): ?>
        <div class="col-md-3">
          <label class="form-label">Category</label>
          <select class="form-select" name="category_id" required>
            <option value="">— Select —</option>
            <?php foreach ($categories as $c): ?>
              <option value="<?=h((int)$c['id'])?>"><?=h($c['name'])?></option>
            <?php endforeach; ?>
          </select>
        </div>
      <?php endif; ?>

      <div class="col-md-4">
        <label class="form-label">Item name</label>
        <input class="form-control" name="name" required placeholder="e.g., Stapler">
      </div>

      <?php if ($hasUnit): ?>
        <div class="col-md-2">
          <label class="form-label">Unit</label>
          <input class="form-control" name="unit" value="pcs">
        </div>
      <?php endif; ?>

      <?php if ($hasThreshold): ?>
        <div class="col-md-2">
          <label class="form-label">Threshold</label>
          <input class="form-control" name="threshold_qty" type="number" min="0" placeholder="e.g., 10">
        </div>
      <?php endif; ?>

      <div class="col-md-2">
        <button class="btn btn-primary w-100">Add</button>
      </div>
    </form>
  </div>
</div>

<?php if ($hasThreshold && count($lowStock) > 0): ?>
  <div class="card shadow-sm mb-3" id="low-stock">
    <div class="card-body">
      <h5 class="mb-3">Low stock items</h5>
      <div class="table-responsive">
        <table class="table table-hover align-middle mb-0">
          <thead class="table-light">
            <tr>
              <?php if ($hasCategories): ?><th style="width:220px;">Category</th><?php endif; ?>
              <th>Item</th>
              <?php if ($hasStock): ?><th style="width:120px;" class="text-end">Stock</th><?php endif; ?>
              <th style="width:140px;" class="text-end">Threshold</th>
            </tr>
          </thead>
          <tbody>
            <?php foreach ($lowStock as $it): ?>
              <tr>
                <?php if ($hasCategories): ?><td class="text-muted"><?=h($it['category_name'] ?? '—')?></td><?php endif; ?>
                <td class="fw-semibold"><?=h($it['name'])?><?php if ($hasUnit): ?> <span class="text-muted">(<?=h($it['unit'])?>)</span><?php endif; ?></td>
                <?php if ($hasStock): ?><td class="text-end fw-semibold"><?=h((int)$it['stock_qty'])?></td><?php endif; ?>
                <td class="text-end text-muted"><?=h((int)$it['threshold_qty'])?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
      <div class="text-muted small mt-2">Tip: Create a purchase to increase stock.</div>
    </div>
  </div>
<?php endif; ?>

<div class="card shadow-sm">
  <div class="card-body p-0">
    <div class="table-responsive">
      <table class="table table-hover align-middle mb-0">
        <thead class="table-light">
          <tr>
            <?php if ($hasCategories): ?><th style="width:220px;">Category</th><?php endif; ?>
            <th>Item</th>
            <?php if ($hasUnit): ?><th style="width:110px;">Unit</th><?php endif; ?>
            <?php if ($hasStock): ?><th style="width:120px;" class="text-end">Stock</th><?php endif; ?>
            <?php if ($hasThreshold): ?><th style="width:140px;" class="text-end">Threshold</th><?php endif; ?>
            <?php if ($hasActive): ?><th style="width:120px;">Status</th><?php endif; ?>
            <th style="width:240px;"></th>
          </tr>
        </thead>
        <tbody>
          <?php foreach ($items as $it): ?>
            <?php
              $isLow = false;
              if ($hasThreshold && $hasStock && $it['threshold_qty'] !== null && (int)$it['threshold_qty'] > 0) {
                $isLow = ((int)$it['stock_qty'] <= (int)$it['threshold_qty']);
              }
            ?>
            <tr class="<?=($isLow?'table-danger':'')?>">
              <?php if ($hasCategories): ?><td class="text-muted"><?=h($it['category_name'] ?? '—')?></td><?php endif; ?>
              <td class="fw-semibold"><?=h($it['name'])?></td>
              <?php if ($hasUnit): ?><td class="text-muted"><?=h($it['unit'])?></td><?php endif; ?>
              <?php if ($hasStock): ?><td class="text-end fw-semibold"><?=h((int)$it['stock_qty'])?></td><?php endif; ?>
              <?php if ($hasThreshold): ?><td class="text-end text-muted"><?=h($it['threshold_qty']===null?'—':(int)$it['threshold_qty'])?></td><?php endif; ?>
              <?php if ($hasActive): ?>
                <td>
                  <?php if ((int)$it['is_active'] === 1): ?>
                    <span class="badge bg-success">Active</span>
                  <?php else: ?>
                    <span class="badge bg-secondary">Inactive</span>
                  <?php endif; ?>
                  <?php if ($isLow): ?><span class="badge bg-danger ms-1">Low</span><?php endif; ?>
                </td>
              <?php endif; ?>
              <td>
                <button class="btn btn-sm btn-outline-primary" type="button" data-bs-toggle="collapse" data-bs-target="#edit<?=h((int)$it['id'])?>">
                  Edit
                </button>
              </td>
            </tr>

            <tr class="collapse" id="edit<?=h((int)$it['id'])?>">
              <td colspan="10">
                <form method="post" class="p-3 bg-light border-top">
                  <input type="hidden" name="action" value="update">
                  <input type="hidden" name="id" value="<?=h((int)$it['id'])?>">

                  <div class="row g-2 align-items-end">
                    <?php if ($hasCategories): ?>
                      <div class="col-md-3">
                        <label class="form-label">Category</label>
                        <select class="form-select" name="category_id" required>
                          <option value="">— Select —</option>
                          <?php foreach ($categories as $c): ?>
                            <option value="<?=h((int)$c['id'])?>" <?=((int)($it['category_id'] ?? 0)===(int)$c['id']?'selected':'')?>>
                              <?=h($c['name'])?>
                            </option>
                          <?php endforeach; ?>
                        </select>
                      </div>
                    <?php endif; ?>

                    <div class="col-md-4">
                      <label class="form-label">Name</label>
                      <input class="form-control" name="name" value="<?=h($it['name'])?>" required>
                    </div>

                    <?php if ($hasUnit): ?>
                      <div class="col-md-2">
                        <label class="form-label">Unit</label>
                        <input class="form-control" name="unit" value="<?=h($it['unit'] ?? '')?>">
                      </div>
                    <?php endif; ?>

                    <?php if ($hasThreshold): ?>
                      <div class="col-md-2">
                        <label class="form-label">Threshold</label>
                        <input class="form-control" name="threshold_qty" type="number" min="0" value="<?=h($it['threshold_qty'] ?? '')?>">
                      </div>
                    <?php endif; ?>

                    <?php if ($hasActive): ?>
                      <div class="col-md-2">
                        <label class="form-label">Active</label>
                        <select class="form-select" name="is_active">
                          <option value="1" <?=((int)($it['is_active'] ?? 1)===1?'selected':'')?>>Yes</option>
                          <option value="0" <?=((int)($it['is_active'] ?? 1)===0?'selected':'')?>>No</option>
                        </select>
                      </div>
                    <?php endif; ?>

                    <div class="col-md-2">
                      <button class="btn btn-success w-100">Save</button>
                    </div>
                  </div>
                </form>
              </td>
            </tr>

          <?php endforeach; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php include __DIR__ . '/../partials/footer.php'; ?>
